<?php

/**
 * Plugin Name: AmirPay Gateway (درگاه پرداخت امیرپی برای ووکامرس)
 * Plugin URI: https://amirpay.top
 * Description: AmirPay payment gateway for WooCommerce — auto-verify, IP lookup, multilingual checkout and support info.
 * Version: 1.4.2
 * Author: AmirPay Technical Team
 * Author URI: https://amirpay.top
 * Text Domain: amirpay-patched
 * Domain Path: /languages
 *
 * Requires at least: 5.0
 * Tested up to: 6.7
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

// Constants
define('AMIRPAY_VERSION', '1.4.2');
define('AMIRPAY_API_URL', 'https://panel.amirpay.top/api/ipg');
define('AMIRPAY_MAX_AMOUNT', 100000000); // 10 million Toman in Rial
define('AMIRPAY_TIMEOUT', 30);

add_action('plugins_loaded', 'amirpay_init_v142', 11);

function amirpay_init_v142()
{
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class WC_Gateway_AmirPay_v142 extends WC_Payment_Gateway
    {
        /**
         * Constructor
         */
        public function __construct()
        {
            $this->id = 'amirpay_v142';
            $this->method_title = __('AmirPay Gateway', 'amirpay-patched');
            $this->method_description = __('AmirPay payment gateway for WooCommerce with auto-verify, IP lookup and multilingual support.', 'amirpay-patched');
            $this->has_fields = false;
            $this->supports = array('products');

            $this->init_form_fields();
            $this->init_settings();

            // Load settings
            $this->enabled = $this->get_option('enabled', 'yes');
            $this->title = $this->get_option('title', 'AmirPay');
            $this->merchant = $this->get_option('merchant', '');
            $this->callback_url = $this->get_option('callback', '');
            $this->default_lang = $this->get_option('default_lang', 'fa');
            $this->enable_lang_switch = 'yes' === $this->get_option('enable_lang_switch', 'yes');
            $this->show_branding = 'yes' === $this->get_option('show_branding', 'yes');
            $this->branding_link = $this->get_option('branding_link', 'https://amirpay.top');
            $this->support_contact = $this->get_option('support_contact', '021-91095537 | amirpay.top');
            $this->debug = 'yes' === $this->get_option('debug', 'yes');

            // Hooks
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_checkout_before_customer_details', array($this, 'render_language_selector_above_checkout'), 5);
            add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
            add_action('wp_ajax_amirpay_get_server_ip', array($this, 'ajax_get_server_ip'));
            add_action('template_redirect', array($this, 'handle_payment_return'), 5);
        }

        /**
         * Initialize form fields
         */
        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => __('Enable/Disable', 'amirpay-patched'),
                    'type' => 'checkbox',
                    'label' => __('Enable AmirPay', 'amirpay-patched'),
                    'default' => 'yes'
                ),
                'title' => array(
                    'title' => __('Title', 'amirpay-patched'),
                    'type' => 'text',
                    'description' => __('Payment method title shown to customers during checkout.', 'amirpay-patched'),
                    'default' => 'AmirPay',
                    'desc_tip' => true
                ),
                'merchant' => array(
                    'title' => __('Merchant ID', 'amirpay-patched'),
                    'type' => 'text',
                    'description' => __('Get your Merchant ID from panel.amirpay.top/fa/dashboard/developers after verification.', 'amirpay-patched'),
                    'default' => '',
                    'desc_tip' => false
                ),
                'callback' => array(
                    'title' => __('Callback URL (optional)', 'amirpay-patched'),
                    'type' => 'text',
                    'description' => __('Leave empty to use default WooCommerce API endpoint. Custom callback URL must be a valid URL.', 'amirpay-patched'),
                    'default' => '',
                    'placeholder' => 'https://yoursite.com/callback',
                    'desc_tip' => false
                ),
                'default_lang' => array(
                    'title' => __('Default Language', 'amirpay-patched'),
                    'type' => 'select',
                    'options' => array(
                        'fa' => __('فارسی (Persian)', 'amirpay-patched'),
                        'en' => __('English', 'amirpay-patched'),
                        'ar' => __('العربية (Arabic)', 'amirpay-patched'),
                        'tr' => __('Türkçe (Turkish)', 'amirpay-patched')
                    ),
                    'description' => __('Default language for payment gateway interface.', 'amirpay-patched'),
                    'default' => 'fa',
                    'desc_tip' => true
                ),
                'enable_lang_switch' => array(
                    'title' => __('Enable Language Switcher', 'amirpay-patched'),
                    'type' => 'checkbox',
                    'label' => __('Show language selector on checkout page', 'amirpay-patched'),
                    'default' => 'yes'
                ),
                'show_branding' => array(
                    'title' => __('Show Branding', 'amirpay-patched'),
                    'type' => 'checkbox',
                    'label' => __('Show "Developed by AmirPay Technical Team" in checkout footer', 'amirpay-patched'),
                    'default' => 'yes'
                ),
                'branding_link' => array(
                    'title' => __('Branding Link', 'amirpay-patched'),
                    'type' => 'text',
                    'description' => __('Link for AmirPay branding (default: https://amirpay.top).', 'amirpay-patched'),
                    'default' => 'https://amirpay.top',
                    'desc_tip' => true
                ),
                'support_contact' => array(
                    'title' => __('Support Contact', 'amirpay-patched'),
                    'type' => 'text',
                    'description' => __('Support contact information displayed in plugin settings.', 'amirpay-patched'),
                    'default' => '021-91095537 | amirpay.top',
                    'desc_tip' => true
                ),
                'debug' => array(
                    'title' => __('Debug Logging', 'amirpay-patched'),
                    'type' => 'checkbox',
                    'label' => __('Enable debug logging. Logs will be written to WooCommerce logs.', 'amirpay-patched'),
                    'default' => 'yes'
                )
            );
        }

        /**
         * Admin options page
         */
        public function admin_options()
        {
?>
            <div class="amirpay-admin-header" style="display:flex;align-items:center;margin-bottom:20px;">
                <?php
                $logo_url = plugins_url('assets/logo.png', __FILE__);
                if (file_exists(plugin_dir_path(__FILE__) . 'assets/logo.png')) {
                    echo '<img src="' . esc_url($logo_url) . '" alt="AmirPay" style="height:48px;margin-right:12px;">';
                }
                ?>
                <div>
                    <h3><?php echo esc_html($this->method_title); ?></h3>
                    <p><?php echo esc_html($this->method_description); ?></p>
                </div>
            </div>
            <?php
            parent::admin_options();
            $this->render_ip_lookup_section();
            ?>
            <div class="amirpay-admin-footer" style="margin-top:20px;padding-top:20px;border-top:1px solid #ddd;">
                <p><strong><?php echo esc_html__('Support:', 'amirpay-patched'); ?></strong>
                    <?php echo esc_html($this->support_contact); ?>
                </p>
                <p style="font-size:12px;color:#666;">
                    <?php echo esc_html__('Get Merchant ID:', 'amirpay-patched'); ?>
                    <a href="https://panel.amirpay.top/fa/dashboard/developers" target="_blank">
                        panel.amirpay.top/fa/dashboard/developers
                    </a>
                </p>
            </div>
        <?php
        }

        /**
         * Render language selector on checkout
         */
        public function render_language_selector_above_checkout()
        {
            if (!is_checkout() || !$this->enable_lang_switch) {
                return;
            }
        ?>
            <div class="amirpay-lang-area" style="margin-bottom:18px;padding:10px;border:1px solid #eee;border-radius:8px;background:#fafafa;">
                <label style="display:block;font-weight:700;margin-bottom:6px;">
                    <?php echo esc_html__('🌐 انتخاب زبان درگاه پرداخت', 'amirpay-patched'); ?>
                </label>
                <p style="margin:0 0 8px;color:#555;">
                    <?php echo esc_html__('زبان مورد نظر خود را برای تجربه پرداخت انتخاب کنید.', 'amirpay-patched'); ?>
                </p>
                <select id="amirpay_lang_checkout" class="amirpay-lang-select-inline"
                    style="padding:6px 8px;border-radius:6px;border:1px solid #ddd;width:100%;max-width:200px;">
                    <option value="fa"><?php echo esc_html__('فارسی', 'amirpay-patched'); ?></option>
                    <option value="en"><?php echo esc_html__('English', 'amirpay-patched'); ?></option>
                    <option value="ar"><?php echo esc_html__('العربية', 'amirpay-patched'); ?></option>
                    <option value="tr"><?php echo esc_html__('Türkçe', 'amirpay-patched'); ?></option>
                </select>
            </div>
            <script>
                (function() {
                    try {
                        var sel = document.getElementById('amirpay_lang_checkout');
                        if (!sel) return;

                        // Load saved language preference
                        try {
                            var savedLang = localStorage.getItem('amirpay_lang');
                            if (savedLang) sel.value = savedLang;
                        } catch (e) {}

                        // Save language preference on change
                        sel.addEventListener('change', function() {
                            try {
                                localStorage.setItem('amirpay_lang', sel.value);
                            } catch (e) {}

                            if (window.amirpayFront && typeof window.amirpayFront.onLangChange === 'function') {
                                window.amirpayFront.onLangChange(sel.value);
                            }
                        });
                    } catch (e) {
                        console.error('AmirPay language selector error:', e);
                    }
                })();
            </script>
        <?php
        }

        /**
         * Enqueue frontend assets
         */
        public function enqueue_frontend_assets()
        {
            if (!is_checkout()) {
                return;
            }

            $script_path = plugins_url('/assets/amirpay-frontend-v142.js', __FILE__);
            wp_register_script('amirpay-frontend', $script_path, array(), AMIRPAY_VERSION, true);

            $translations = array(
                'fa' => array('branding' => 'توسعه و اجرا توسط تیم فنی AmirPay | AmirPay.top'),
                'en' => array('branding' => 'Developed by AmirPay Technical Team | AmirPay.top'),
                'ar' => array('branding' => 'تم التطوير بواسطة فريق AmirPay التقني | AmirPay.top'),
                'tr' => array('branding' => 'AmirPay Teknik Ekibi tarafından geliştirildi | AmirPay.top')
            );

            wp_localize_script('amirpay-frontend', 'amirpayFront', array(
                'defaultLang' => $this->default_lang,
                'enableSwitcher' => $this->enable_lang_switch,
                'showBranding' => $this->show_branding,
                'brandingLink' => esc_url($this->branding_link),
                'translations' => $translations
            ));

            wp_enqueue_script('amirpay-frontend');
            wp_add_inline_style(
                'wp-block-library',
                '.amirpay-branding{color:#666;font-size:13px;margin-top:12px;text-align:left}'
            );
        }

        /**
         * Enqueue admin assets
         */
        public function enqueue_admin_assets()
        {
            // Admin assets if needed in future
        }

        /**
         * Render IP lookup section in admin
         */
        public function render_ip_lookup_section()
        {
            $msgs = array(
                'title' => __('Server IP Lookup', 'amirpay-patched'),
                'label' => __('Your Server IP:', 'amirpay-patched'),
                'hint' => __('Add this IP to the allowed IPs section in panel.amirpay.top', 'amirpay-patched'),
                'refresh' => __('Refresh IP', 'amirpay-patched'),
                'copy' => __('Copy IP', 'amirpay-patched'),
                'copied' => __('IP Copied!', 'amirpay-patched')
            );
        ?>
            <hr style="margin:20px 0;" />
            <h4><?php echo esc_html($msgs['title']); ?></h4>
            <p>
                <strong><?php echo esc_html($msgs['label']); ?></strong>
                <input type="text" id="amirpay_server_ip" readonly
                    style="width:220px;margin:0 8px;padding:4px 8px;border:1px solid #ddd;border-radius:4px;"
                    value="">
                <button type="button" class="button" id="amirpay_refresh_ip">
                    <?php echo esc_html($msgs['refresh']); ?>
                </button>
                <button type="button" class="button" id="amirpay_copy_ip">
                    <?php echo esc_html($msgs['copy']); ?>
                </button>
            </p>
            <p style="color:#666;font-size:13px;">
                <?php echo esc_html($msgs['hint']); ?>
            </p>
            <script>
                (function() {
                    var ajaxUrl = '<?php echo esc_js(admin_url('admin-ajax.php')); ?>';
                    var copyBtn = document.getElementById('amirpay_copy_ip');
                    var refreshBtn = document.getElementById('amirpay_refresh_ip');
                    var ipInput = document.getElementById('amirpay_server_ip');
                    var originalCopyText = '<?php echo esc_js($msgs['copy']); ?>';

                    function setIP(ip) {
                        if (ipInput && ip) {
                            ipInput.value = ip;
                        }
                    }

                    function fetchIP() {
                        if (!refreshBtn || !ipInput) return;

                        refreshBtn.disabled = true;
                        refreshBtn.textContent = '<?php echo esc_js($msgs['refresh']); ?>...';

                        var xhr = new XMLHttpRequest();
                        xhr.open('POST', ajaxUrl, true);
                        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded; charset=UTF-8');

                        xhr.onload = function() {
                            refreshBtn.disabled = false;
                            refreshBtn.textContent = '<?php echo esc_js($msgs['refresh']); ?>';

                            if (xhr.status === 200) {
                                try {
                                    var response = JSON.parse(xhr.responseText);
                                    if (response.success && response.data && response.data.ip) {
                                        setIP(response.data.ip);
                                    }
                                } catch (e) {
                                    console.error('Error parsing IP response:', e);
                                }
                            }
                        };

                        xhr.onerror = function() {
                            refreshBtn.disabled = false;
                            refreshBtn.textContent = '<?php echo esc_js($msgs['refresh']); ?>';
                        };

                        xhr.send('action=amirpay_get_server_ip');
                    }

                    if (copyBtn) {
                        copyBtn.addEventListener('click', function() {
                            if (ipInput && ipInput.value) {
                                ipInput.select();
                                ipInput.setSelectionRange(0, 99999); // For mobile

                                try {
                                    document.execCommand('copy');
                                    copyBtn.textContent = '<?php echo esc_js($msgs['copied']); ?>';
                                    setTimeout(function() {
                                        copyBtn.textContent = originalCopyText;
                                    }, 2000);
                                } catch (e) {
                                    console.error('Copy failed:', e);
                                }
                            }
                        });
                    }

                    if (refreshBtn) {
                        refreshBtn.addEventListener('click', fetchIP);
                    }

                    // Auto-fetch on load
                    fetchIP();
                })();
            </script>
<?php
        }

        /**
         * AJAX handler for server IP lookup
         */
        public function ajax_get_server_ip()
        {
            // Verify this is an AJAX request from admin
            if (!current_user_can('manage_woocommerce')) {
                wp_send_json_error(array('message' => __('Unauthorized', 'amirpay-patched')));
                return;
            }

            $ip = $this->get_server_ip();
            wp_send_json_success(array('ip' => $ip));
        }

        /**
         * Get server IP address
         *
         * @return string Server IP address
         */
        private function get_server_ip()
        {
            // Try SERVER_ADDR first
            $ip = isset($_SERVER['SERVER_ADDR']) ? sanitize_text_field($_SERVER['SERVER_ADDR']) : '';

            // Try gethostbyname if SERVER_ADDR is empty
            if (empty($ip)) {
                $hostname = function_exists('gethostname') ? gethostname() : '';
                if (!empty($hostname)) {
                    $ip = gethostbyname($hostname);
                    // If hostname returned itself, it's invalid
                    if ($ip === $hostname) {
                        $ip = '';
                    }
                }
            }

            // Fallback to external service
            if (empty($ip) || !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                $ip = $this->get_external_ip();
            }

            return $ip ? $ip : __('Unable to determine', 'amirpay-patched');
        }

        /**
         * Get IP from external service
         *
         * @return string IP address or empty string
         */
        private function get_external_ip()
        {
            if (!function_exists('curl_init')) {
                return '';
            }

            $ch = curl_init('https://api.ipify.org?format=json');
            curl_setopt_array($ch, array(
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 5,
                CURLOPT_CONNECTTIMEOUT => 5,
                CURLOPT_SSL_VERIFYPEER => true
            ));

            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($http_code === 200 && $response) {
                $data = json_decode($response, true);
                if (isset($data['ip']) && filter_var($data['ip'], FILTER_VALIDATE_IP)) {
                    return sanitize_text_field($data['ip']);
                }
            }

            return '';
        }

        /**
         * Process payment
         *
         * @param int $order_id Order ID
         * @return array Payment result
         */
        public function process_payment($order_id)
        {
            $order = wc_get_order($order_id);
            if (!$order) {
                $this->log_error('Order not found: ' . $order_id);
                wc_add_notice(__('Order not found.', 'amirpay-patched'), 'error');
                return array('result' => 'fail', 'redirect' => '');
            }

            // Validate and prepare payment data
            $payment_data = $this->prepare_payment_data($order);
            if (is_wp_error($payment_data)) {
                wc_add_notice($payment_data->get_error_message(), 'error');
                return array('result' => 'fail', 'redirect' => '');
            }

            // Send payment request
            $response = $this->send_payment_request($payment_data);
            if (is_wp_error($response)) {
                $this->log_error('Payment request failed: ' . $response->get_error_message());
                wc_add_notice(__('Payment gateway connection error.', 'amirpay-patched'), 'error');
                return array('result' => 'fail', 'redirect' => '');
            }

            // Process response
            $result = $this->process_payment_response($response, $order);
            return $result;
        }

        /**
         * Prepare payment data
         *
         * @param WC_Order $order Order object
         * @return array|WP_Error Payment data or error
         */
        private function prepare_payment_data($order)
        {
            // Validate merchant
            $merchant = $this->sanitize_merchant_id($this->merchant);
            if (empty($merchant)) {
                return new WP_Error('invalid_merchant', __('Merchant ID is not configured properly.', 'amirpay-patched'));
            }

            // Validate and calculate amount
            $amount = $this->calculate_amount($order);
            if (is_wp_error($amount)) {
                return $amount;
            }

            // Get callback URL
            $callback_url = $this->get_callback_url();
            if (is_wp_error($callback_url)) {
                return $callback_url;
            }

            // Prepare description
            $description = $this->prepare_description($order);

            return array(
                'merchant' => $merchant,
                'amount' => $amount,
                'callbackUrl' => $callback_url,
                'desc' => $description
            );
        }

        /**
         * Sanitize merchant ID
         *
         * @param string $merchant Merchant ID
         * @return string Sanitized merchant ID
         */
        private function sanitize_merchant_id($merchant)
        {
            $merchant = sanitize_text_field($merchant);
            // Only allow alphanumeric, dash, and underscore
            return preg_replace('/[^a-zA-Z0-9_-]/', '', $merchant);
        }

        /**
         * Calculate and validate amount
         *
         * @param WC_Order $order Order object
         * @return int|WP_Error Amount in Rial or error
         */
        private function calculate_amount($order)
        {
            $order_total = floatval($order->get_total());
            $amount = (int)($order_total * 10); // Convert to Rial (Toman * 10)

            if ($amount <= 0) {
                $this->log_error('Invalid amount: ' . $amount . ' (Order total: ' . $order_total . ')');
                return new WP_Error('invalid_amount', __('Invalid order amount.', 'amirpay-patched'));
            }

            if ($amount > AMIRPAY_MAX_AMOUNT) {
                $max_toman = number_format(AMIRPAY_MAX_AMOUNT / 10, 0);
                $this->log_error('Amount too large: ' . $amount . ' (Order total: ' . $order_total . ')');
                return new WP_Error(
                    'amount_too_large',
                    sprintf(__('Order amount is too large. Maximum amount is %s Toman.', 'amirpay-patched'), $max_toman)
                );
            }

            return $amount;
        }

        /**
         * Get and validate callback URL
         *
         * @return string|WP_Error Callback URL or error
         */
        private function get_callback_url()
        {
            $callback_url = $this->callback_url;

            // Use default WooCommerce API endpoint if empty
            if (empty($callback_url)) {
                $callback_url = WC()->api_request_url('WC_Gateway_AmirPay_v142');
            }

            // Sanitize and validate
            $callback_url = esc_url_raw($callback_url);
            if (!filter_var($callback_url, FILTER_VALIDATE_URL)) {
                $this->log_error('Invalid callback URL: ' . $callback_url);
                return new WP_Error('invalid_callback', __('Invalid callback URL configuration.', 'amirpay-patched'));
            }

            return $callback_url;
        }

        /**
         * Prepare order description
         *
         * @param WC_Order $order Order object
         * @return string Description
         */
        private function prepare_description($order)
        {
            $order_number = $order->get_order_number();
            // Remove special characters, keep only alphanumeric, spaces, and dashes
            $order_number = preg_replace('/[^a-zA-Z0-9\s-]/', '', $order_number);
            $description = sprintf(__('Order %s', 'amirpay-patched'), $order_number);
            return sanitize_text_field($description);
        }

        /**
         * Send payment request to API
         *
         * @param array $data Payment data
         * @return array|WP_Error Response or error
         */
        private function send_payment_request($data)
        {
            $api_url = AMIRPAY_API_URL . '/start';

            if ($this->debug) {
                $this->log('Sending payment request to: ' . $api_url);
                $this->log('Request data: ' . wp_json_encode($data, JSON_UNESCAPED_UNICODE));
            }

            $response = wp_remote_post($api_url, array(
                'body' => wp_json_encode($data, JSON_UNESCAPED_UNICODE),
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ),
                'timeout' => AMIRPAY_TIMEOUT,
                'sslverify' => true,
                'redirection' => 10,
                'httpversion' => '1.1'
            ));

            return $response;
        }

        /**
         * Process payment response
         *
         * @param array $response HTTP response
         * @param WC_Order $order Order object
         * @return array Payment result
         */
        private function process_payment_response($response, $order)
        {
            $http_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $result = json_decode($body, true);

            if ($this->debug) {
                $this->log('Response HTTP Code: ' . $http_code);
                $this->log('Response Body: ' . $body);
            }

            // Check HTTP error
            if ($http_code !== 200) {
                $error_msg = isset($result['message']) ? $result['message'] :
                    sprintf(__('HTTP Error: %d', 'amirpay-patched'), $http_code);
                $this->log_error('HTTP Error: ' . $http_code . ' - ' . $body);
                wc_add_notice(__('Payment gateway error: ', 'amirpay-patched') . $error_msg, 'error');
                return array('result' => 'fail', 'redirect' => '');
            }

            // Check JSON error
            if (isset($result['success']) && $result['success'] === false) {
                return $this->handle_payment_error($result, $body);
            }

            // Check success response
            if (isset($result['success']) && $result['success'] === true && isset($result['data']['linkMain'])) {
                return $this->handle_payment_success($result, $order);
            }

            // Invalid response structure
            $this->log_error('Invalid response structure: ' . $body);
            $error_msg = isset($result['message']) ? $result['message'] :
                __('Invalid response from payment gateway.', 'amirpay-patched');
            wc_add_notice(__('Payment gateway error: ', 'amirpay-patched') . $error_msg, 'error');
            return array('result' => 'fail', 'redirect' => '');
        }

        /**
         * Handle payment error
         *
         * @param array $result Error result
         * @param string $body Response body
         * @return array Payment result
         */
        private function handle_payment_error($result, $body)
        {
            $error_msg = isset($result['message']) ? $result['message'] :
                __('Unknown error from payment gateway.', 'amirpay-patched');

            // Add specific help for IP restriction error
            if (
                stripos($error_msg, 'IP') !== false ||
                stripos($error_msg, 'آدرس') !== false ||
                stripos($error_msg, 'مجاز') !== false
            ) {
                $error_msg .= ' ' . __('Please go to WooCommerce > Settings > Payments > AmirPay Gateway and copy your server IP from the IP lookup section, then add it to panel.amirpay.top in the allowed IPs section.', 'amirpay-patched');
            }

            // Add help for invalid data error
            if (stripos($error_msg, 'نامعتبر') !== false || stripos($error_msg, 'invalid') !== false) {
                $error_msg .= ' ' . __('Please check: 1) Amount is within limits (max 10M Toman), 2) No special characters in order number, 3) Callback URL matches registered domain.', 'amirpay-patched');
            }

            $this->log_error('Payment Error: ' . $error_msg);
            $this->log_error('Full Error Response: ' . $body);

            wc_add_notice(__('Payment gateway error: ', 'amirpay-patched') . $error_msg, 'error');
            return array('result' => 'fail', 'redirect' => '');
        }

        /**
         * Handle payment success
         *
         * @param array $result Success result
         * @param WC_Order $order Order object
         * @return array Payment result
         */
        private function handle_payment_success($result, $order)
        {
            $linkMain = esc_url_raw($result['data']['linkMain']);
            $orderId = isset($result['data']['orderId']) ? sanitize_text_field($result['data']['orderId']) : '';

            // Save transaction ID
            if (!empty($orderId)) {
                $order->update_meta_data('_amirpay_transaction_id', $orderId);
                $order->save();
            }

            if ($this->debug) {
                $this->log('Payment successful. Redirecting to: ' . $linkMain);
            }

            return array('result' => 'success', 'redirect' => $linkMain);
        }

        /**
         * Handle payment return from gateway
         */
        public function handle_payment_return()
        {
            if (!isset($_GET['wc-api']) || $_GET['wc-api'] !== 'WC_Gateway_AmirPay_v142') {
                return;
            }

            // Sanitize input
            $transaction_id = isset($_GET['transaction_id']) ? sanitize_text_field($_GET['transaction_id']) : '';
            $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
            $order_id = isset($_GET['order_id']) ? absint($_GET['order_id']) : 0;

            if (empty($transaction_id) || empty($order_id)) {
                wc_add_notice(__('Invalid payment return parameters.', 'amirpay-patched'), 'error');
                wp_safe_redirect(wc_get_checkout_url());
                exit;
            }

            $order = wc_get_order($order_id);
            if (!$order) {
                wc_add_notice(__('Order not found.', 'amirpay-patched'), 'error');
                wp_safe_redirect(wc_get_checkout_url());
                exit;
            }

            if ($status === 'success') {
                $this->verify_payment($transaction_id, $order_id, $order);
            } else {
                wc_add_notice(__('Payment verification failed.', 'amirpay-patched'), 'error');
                wp_safe_redirect($order->get_checkout_order_received_url());
                exit;
            }
        }

        /**
         * Verify payment with gateway
         *
         * @param string $transaction_id Transaction ID
         * @param int $order_id Order ID
         * @param WC_Order $order Order object
         */
        private function verify_payment($transaction_id, $order_id, $order)
        {
            $merchant = $this->sanitize_merchant_id($this->merchant);
            if (empty($merchant)) {
                $this->log_error('Cannot verify: Merchant ID is empty');
                wc_add_notice(__('Payment verification error.', 'amirpay-patched'), 'error');
                wp_safe_redirect($order->get_checkout_order_received_url());
                exit;
            }

            $verify_url = AMIRPAY_API_URL . '/verify';
            $data = array(
                'merchant' => $merchant,
                'transaction_id' => $transaction_id,
                'order_id' => $order_id
            );

            if ($this->debug) {
                $this->log('Verifying payment: ' . wp_json_encode($data, JSON_UNESCAPED_UNICODE));
            }

            $response = wp_remote_post($verify_url, array(
                'body' => wp_json_encode($data, JSON_UNESCAPED_UNICODE),
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ),
                'timeout' => AMIRPAY_TIMEOUT,
                'sslverify' => true,
                'redirection' => 10,
                'httpversion' => '1.1'
            ));

            if (is_wp_error($response)) {
                $this->log_error('Verify Error: ' . $response->get_error_message());
                wc_add_notice(__('Payment verification error.', 'amirpay-patched'), 'error');
                wp_safe_redirect($order->get_checkout_order_received_url());
                exit;
            }

            $body = wp_remote_retrieve_body($response);
            $result = json_decode($body, true);

            if ($this->debug) {
                $this->log('Verify Response: ' . $body);
            }

            if (isset($result['success']) && $result['success'] === true) {
                // Complete payment if not already completed
                if (!in_array($order->get_status(), array('completed', 'processing'), true)) {
                    $order->payment_complete($transaction_id);
                    $order->add_meta_data('_amirpay_paid', current_time('mysql'));
                    $order->save();

                    if ($this->debug) {
                        $this->log('Payment completed for order: ' . $order_id);
                    }
                }

                wp_safe_redirect($order->get_checkout_order_received_url());
                exit;
            }

            // Verification failed
            $this->log_error('Payment verification failed: ' . $body);
            wc_add_notice(__('Payment verification failed.', 'amirpay-patched'), 'error');
            wp_safe_redirect($order->get_checkout_order_received_url());
            exit;
        }

        /**
         * Log error message
         *
         * @param string $message Error message
         */
        private function log_error($message)
        {
            if ($this->debug && function_exists('wc_get_logger')) {
                $logger = wc_get_logger();
                $logger->error($message, array('source' => 'amirpay_v142'));
            }
        }

        /**
         * Log debug message
         *
         * @param string $message Debug message
         */
        private function log($message)
        {
            if ($this->debug && function_exists('wc_get_logger')) {
                $logger = wc_get_logger();
                $logger->debug($message, array('source' => 'amirpay_v142'));
            }
        }
    }

    /**
     * Add gateway to WooCommerce
     *
     * @param array $methods Payment methods
     * @return array Updated payment methods
     */
    function woocommerce_add_amirpay_gateway($methods)
    {
        $methods[] = 'WC_Gateway_AmirPay_v142';
        return $methods;
    }

    add_filter('woocommerce_payment_gateways', 'woocommerce_add_amirpay_gateway');
}
